/* TROIA distributor
 * version 0.4 date 2005-01-26
 * Copyright (C) 2003-2005 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <sys/types.h>
#include <netinet/in.h>

#include "td_fmt.h"
#include "protocols.h"

//maximum parameters of protocols
#define height_max 1000
#define width_max 1000
#define channels_max 3
#define colors_max 256

//protocol magics
#define BLP_MAGIC "\xDE""\xAD""\xBE""\xEF"
#define BLP_MAGIC_LEN 4
#define EBLP_MAGIC "\xFE""\xED""\xBE""\xEF"
#define EBLP_MAGIC_LEN 4
#define MCUF_MAGIC "\x23""\x54""\x26""\x66"
#define MCUF_MAGIC_LEN 4

//protocol headers
typedef struct s_proto_blp_hdr
{
  uint32_t magic;
  uint32_t frame_no;
  uint16_t width;
  uint16_t height;
} st_proto_blp_hdr;
#define proto_blp_magic 0xDEADBEEF
typedef struct s_proto_eblp_hdr
{
  uint32_t magic;
  uint32_t frame_no;
  uint16_t width;
  uint16_t height;
} st_proto_eblp_hdr;
#define proto_eblp_magic 0xFEEDBEEF
typedef struct s_proto_mcuf_hdr
{
  uint32_t magic;
  uint16_t height;
  uint16_t width;
  uint16_t channels;
  uint16_t maxval;
} st_proto_mcuf_hdr;
#define proto_mcuf_magic 0x23542666

//identify the protocol by the magic
//returns 1 on success, 0 on error
static int proto_identify( unsigned char * p_datagram, unsigned int datagram_len,
                           unsigned short * p_height, unsigned short * p_width,
			   unsigned short * p_channels, unsigned short * p_colors,
			   unsigned char * * pp_data )
{
  //BLP
  if( datagram_len >= sizeof( st_proto_blp_hdr ) && ntohl( ((st_proto_blp_hdr *)p_datagram)->magic ) == proto_blp_magic )
  {
    //*p_height
    *p_height = ntohs( ((st_proto_blp_hdr *)p_datagram)->height );
    if( *p_height == 0 || *p_height > height_max )
      return 0;
    //*p_width
    *p_width = ntohs( ((st_proto_blp_hdr *)p_datagram)->width );
    if( *p_width == 0 || *p_width > width_max )
      return 0;
    //*p_channels: 1
    *p_channels = 1;
    //*p_colors: 2
    *p_colors = 2;

    //check size of message
    if( datagram_len < sizeof( st_proto_blp_hdr ) + *p_height * *p_width * *p_channels )
      return 0;

    //set pointer to real data
    *pp_data = p_datagram + sizeof( st_proto_blp_hdr );

    return 1;
  }
  
  //EBLP
  else if( datagram_len >= sizeof( st_proto_eblp_hdr ) && ntohl( ((st_proto_eblp_hdr *)p_datagram)->magic ) == proto_eblp_magic )
  {
    //*p_height
    *p_height = ntohs( ((st_proto_blp_hdr *)p_datagram)->height );
    if( *p_height == 0 || *p_height > height_max )
      return 0;
    //*p_width
    *p_width = ntohs( ((st_proto_blp_hdr *)p_datagram)->width );
    if( *p_width == 0 || *p_width > width_max )
      return 0;
    //*p_channels: 1
    *p_channels = 1;
    //*p_colors: 256
    *p_colors = 256;

    //check size of message
    if( datagram_len < sizeof( st_proto_eblp_hdr ) + *p_height * *p_width * *p_channels )
      return 0;

    //set pointer to real data
    *pp_data = p_datagram + sizeof( st_proto_eblp_hdr );

    return 1;
  }
  
  //MCUF
  else if( datagram_len >= sizeof( st_proto_mcuf_hdr ) && ntohl( ((st_proto_mcuf_hdr *)p_datagram)->magic ) == proto_mcuf_magic )
  {
    //*p_height
    *p_height = ntohs( ((st_proto_mcuf_hdr *)p_datagram)->height );
    if( *p_height == 0 || *p_height > height_max )
      return 0;
    //*p_width
    *p_width = ntohs( ((st_proto_mcuf_hdr *)p_datagram)->width );
    if( *p_width == 0 || *p_width > width_max )
      return 0;
    //*p_channels
    *p_channels = ntohs( ((st_proto_mcuf_hdr *)p_datagram)->channels );
    if( *p_channels == 0 || *p_channels > channels_max )
      return 0;
    //*p_colors
    *p_colors = ntohs( ((st_proto_mcuf_hdr *)p_datagram)->maxval ) + 1;
    if( *p_colors <= 1 || *p_colors > colors_max )
      return 0;

    //check size of message
    if( datagram_len < sizeof( st_proto_mcuf_hdr ) + *p_height * *p_width * *p_channels )
      return 0;

    //set pointer to real data
    *pp_data = p_datagram + sizeof( st_proto_blp_hdr );

    return 1;
  }

  //unknown protocol
  else
    return 0;
}

//put pixels into output buffer - 1 channel 2 colors version
static void proto_out_chan1col2( unsigned short height, unsigned short width,
                                 unsigned char * p_data, st_td_fmt * p_td_fmt,
                                 unsigned char * * p_out_buffers )
{
  unsigned short ignore_top, ignore_left, ignore_i;
  unsigned short free_top, free_left, size_x, size_y;
  unsigned short x, y, xx, yy;
  unsigned int i;
  st_td_fmt_pix * p_td_fmt_pix;

  //calculate parameters for conversion
  if( height > p_td_fmt->height )
  {
    ignore_top = (height - p_td_fmt->height) / 2;
    free_top = 0;
    size_y = p_td_fmt->height;
  }
  else
  {
    ignore_top = 0;
    free_top = (p_td_fmt->height - height) / 2;
    size_y = height;
  }
  if( width > p_td_fmt->width )
  {
    ignore_left = (width - p_td_fmt->width) / 2;
    ignore_i = width - p_td_fmt->width;
    free_left = 0;
    size_x = p_td_fmt->width;
  }
  else
  {
    ignore_left = 0;
    ignore_i = 0;
    free_left = (p_td_fmt->width - width) / 2;
    size_x = width;
  }

  //write pixels into output buffer
  i = ignore_top * width + ignore_left;
  for( y = 0, yy = free_top; y < size_y; y++, yy++, i += ignore_i )
  {
    for( x = 0, xx = free_left; x < size_x; x++, xx++, i++ )
    {
      p_td_fmt_pix = &p_td_fmt->pixels[yy][xx];
      if( p_td_fmt_pix->valid )
        p_out_buffers[p_td_fmt_pix->dev_no][p_td_fmt_pix->out_buf_ofs]
          = (p_data[i] ? 0x7F : 0x00) | p_td_fmt_pix->sync_bit;
    }
  }
}

//put pixels into output buffer - 1 channel 256 colors version
static void proto_out_chan1col256( unsigned short height, unsigned short width,
                                   unsigned char * p_data, st_td_fmt * p_td_fmt,
                                   unsigned char * * p_out_buffers )
{
  unsigned short ignore_top, ignore_left, ignore_i;
  unsigned short free_top, free_left, size_x, size_y;
  unsigned short x, y, xx, yy;
  unsigned int i;
  st_td_fmt_pix * p_td_fmt_pix;

  //calculate parameters for conversion
  if( height > p_td_fmt->height )
  {
    ignore_top = (height - p_td_fmt->height) / 2;
    free_top = 0;
    size_y = p_td_fmt->height;
  }
  else
  {
    ignore_top = 0;
    free_top = (p_td_fmt->height - height) / 2;
    size_y = height;
  }
  if( width > p_td_fmt->width )
  {
    ignore_left = (width - p_td_fmt->width) / 2;
    ignore_i = width - p_td_fmt->width;
    free_left = 0;
    size_x = p_td_fmt->width;
  }
  else
  {
    ignore_left = 0;
    ignore_i = 0;
    free_left = (p_td_fmt->width - width) / 2;
    size_x = width;
  }

  //write pixels into output buffer
  i = ignore_top * width + ignore_left;
  for( y = 0, yy = free_top; y < size_y; y++, yy++, i += ignore_i )
  {
    for( x = 0, xx = free_left; x < size_x; x++, xx++, i++ )
    {
      p_td_fmt_pix = &p_td_fmt->pixels[yy][xx];
      if( p_td_fmt_pix->valid )
        p_out_buffers[p_td_fmt_pix->dev_no][p_td_fmt_pix->out_buf_ofs]
          = p_data[i] >> 1 | p_td_fmt_pix->sync_bit;
    }
  }
}

//put pixels into output buffer - generic version
static void proto_out_generic( unsigned short height, unsigned short width,
                               unsigned short channels, unsigned short colors,
                               unsigned char * p_data, st_td_fmt * p_td_fmt,
                               unsigned char * * p_out_buffers )
{
  unsigned short ignore_top, ignore_left, ignore_i;
  unsigned short free_top, free_left, size_x, size_y;
  unsigned short x, y, c, xx, yy;
  unsigned int i, value, value_div;
  st_td_fmt_pix * p_td_fmt_pix;

  //calculate parameters for conversion
  if( height > p_td_fmt->height )
  {
    ignore_top = (height - p_td_fmt->height) / 2;
    free_top = 0;
    size_y = p_td_fmt->height;
  }
  else
  {
    ignore_top = 0;
    free_top = (p_td_fmt->height - height) / 2;
    size_y = height;
  }
  if( width > p_td_fmt->width )
  {
    ignore_left = (width - p_td_fmt->width) / 2;
    ignore_i = (width - p_td_fmt->width) * channels;
    free_left = 0;
    size_x = p_td_fmt->width;
  }
  else
  {
    ignore_left = 0;
    ignore_i = 0;
    free_left = (p_td_fmt->width - width) / 2;
    size_x = width;
  }

  //write pixels into output buffer
  i = (ignore_top * width + ignore_left) * channels;
  value_div = channels * colors;
  for( y = 0, yy = free_top; y < size_y; y++, yy++, i += ignore_i )
  {
    for( x = 0, xx = free_left; x < size_x; x++, xx++ )
    {
      p_td_fmt_pix = &p_td_fmt->pixels[yy][xx];
      if( p_td_fmt_pix->valid )
      {
        value = 0;
        for( c = 0; c < channels; c++, i++ )
          if( p_data[i] < colors )
            value += p_data[i];
          else
            value += colors - 1;
        value = ((value << 8) - value) / value_div; //(value * 255) / (channels * colors)
        p_out_buffers[p_td_fmt_pix->dev_no][p_td_fmt_pix->out_buf_ofs]
          = (unsigned char)value >> 1 | p_td_fmt_pix->sync_bit;
      }
    }
  }
}

//processs a recevied datagram
//p_out_buffers must point to an array of dev_cnt output buffers
//  every output buffer must be of size pix_cnt * ser_cnt
//  every pixel in output buffer is encoded with 7 bit
//    sync bit (D7) is set for first pixels of serial ports
//returns 1 on success, 0 on error
int proto_datagram( unsigned char * p_datagram, unsigned int datagram_len,
                    st_td_fmt * p_td_fmt,
                    unsigned char * * p_out_buffers )
{
  unsigned short height, width, channels, colors;
  unsigned char * p_data;

  //identify the protocol by the magic
  if( ! proto_identify( p_datagram, datagram_len, &height, &width, &channels, &colors, &p_data ) )
    return 0;

  //put pixels into output buffer using the appropriate procedure
  if( channels == 1 && colors == 2 )
    proto_out_chan1col2( height, width, p_data, p_td_fmt, p_out_buffers );
  else if( channels == 1 && colors == 256 )
    proto_out_chan1col256( height, width, p_data, p_td_fmt, p_out_buffers );
  else
    proto_out_generic( height, width, channels, colors, p_data, p_td_fmt, p_out_buffers );

  return 1;
}
