/* TROIA distributor
 * version 0.4 date 2005-01-26
 * Copyright (C) 2003-2005 Stefan Schuermans <1stein@schuermans.info>
 * Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <malloc.h>
#include <signal.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <netinet/in.h>

#include "protocols.h"
#include "td_config.h"
#include "td_fmt.h"

//global variables
int end = 0; //set to 1 by signal handler to indicate end of program

//signal handler to end program
static void end_signal( int sig_no )
{
  //end program
  end = 1;
  //keep compiler happy
  sig_no = 0;
}

//main program
int main( int arg_cnt, char * * args )
{
  char * p_config_file;
  unsigned int in_buffer_len, out_buffer_len;
  unsigned char * p_in_buffer;
  unsigned char * * p_out_buffers;
  st_td_fmt * p_td_fmt;
  int in_sock_fd, out_sock_fd;
  struct sockaddr_in addr;
  fd_set fdset;
  struct timeval timeout;
  unsigned short dev_no, ser_no, pix_no;
  int cnt, len, addr_len, i;
  unsigned char * ptr;
  in_addr_t in_addr;

  //print message
  printf( "\n"
          "TROIA distributor\n"
          "version 0.4 date 2005-01-26\n"
          "Copyright (C) 2003-2005 Stefan Schuermans <1stein@schuermans.info>\n"
          "Copyleft: GNU public license - http://www.gnu.org/copyleft/gpl.html\n"
          "\n" );

  //no config file supplied
  if( arg_cnt < 2 )
  {
    printf( "no config file supplied (syntax: \"troia_dist <config-file>\"), using \"troia_dist.conf\"\n" );
    p_config_file = "troia_dist.conf";
  }
  //config file supplied
  else
    p_config_file = args[1];

  //read config file and get settings
  td_config_get( p_config_file );
  printf( "configuration read: %d devices with %d serial ports with %d pixels\n",
          td_out_dev_cnt, td_out_ser_cnt, td_out_pix_cnt );

  //allocate input buffer
  in_buffer_len = 65536; //maximum size of a datagram
  p_in_buffer = (unsigned char *)malloc( in_buffer_len );
  if( p_in_buffer == NULL )
  {
    printf( "could not allocate buffer (input) of size %d\n\n", in_buffer_len );
    return -1;
  }

  //allocate output buffer
  out_buffer_len = td_out_pix_cnt * td_out_ser_cnt; //length of output buffer for 1 device
  len = td_out_dev_cnt * sizeof( unsigned char * ) //dev_cnt pointers to 1D arrays
      + td_out_dev_cnt * out_buffer_len; //dev_cnt output buffers
  p_out_buffers = (unsigned char * *)malloc( len );
  if( p_out_buffers == NULL )
  {
    free( p_in_buffer );
    printf( "could not allocate buffer (output) of size %d\n\n", len );
    return -1;
  }
  //initialize output buffer array structure
  ptr = (unsigned char *)p_out_buffers + td_out_dev_cnt * sizeof( unsigned char * ); //array of pointers to 1D arrays
  for( dev_no = 0; dev_no < td_out_dev_cnt; dev_no++ ) //output buffers
  {
    p_out_buffers[dev_no] = ptr;
    ptr += out_buffer_len;
  }
  //initialize data in output buffer
  for( dev_no = 0; dev_no < td_out_dev_cnt; dev_no++ )
    for( pix_no = 0, i = 0; pix_no < td_out_pix_cnt; pix_no++ )
      for( ser_no = 0; ser_no < td_out_ser_cnt; ser_no++, i++ )
        p_out_buffers[dev_no][i] = pix_no == 0 ? 0x80 : 0x00; //set sync bit in first pixels

  //generate format from format file
  p_td_fmt = td_fmt_load( td_fmt_file );
  if( p_td_fmt == NULL )
  {
    free( p_out_buffers );
    free( p_in_buffer );
    printf( "could not create format from \"%s\"\n\n", td_fmt_file );
    return -1;
  }
  printf( "format file read: %dx%d pixels\n", p_td_fmt->width, p_td_fmt->height );

  //create input socket
  in_sock_fd = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP );
  if( in_sock_fd == -1 )
  {
    td_fmt_free( p_td_fmt );
    free( p_out_buffers );
    free( p_in_buffer );
    printf( "could not create input socket: error: %s\n\n", strerror( errno ) );
    return -1;
  }

  //bind input socket
  addr.sin_family = AF_INET;
  addr.sin_port = htons( td_in_listen_port );
  addr.sin_addr.s_addr = td_in_listen_addr;
  if( bind( in_sock_fd, (struct sockaddr *)&addr, sizeof( addr ) ) == -1 )
  {
    close( in_sock_fd );
    td_fmt_free( p_td_fmt );
    free( p_out_buffers );
    free( p_in_buffer );
    printf( "could not bind input socket to \"%d.%d.%d.%d:%d\" (udp): error: %s\n\n",
      ((unsigned char *)&addr.sin_addr.s_addr)[0],
      ((unsigned char *)&addr.sin_addr.s_addr)[1],
      ((unsigned char *)&addr.sin_addr.s_addr)[2],
      ((unsigned char *)&addr.sin_addr.s_addr)[3],
      ntohs( addr.sin_port ),
      strerror( errno ) );
    return -1;
  }
  printf( "input socket listening on \"%d.%d.%d.%d:%d\" (udp)\n",
    ((unsigned char *)&addr.sin_addr.s_addr)[0],
    ((unsigned char *)&addr.sin_addr.s_addr)[1],
    ((unsigned char *)&addr.sin_addr.s_addr)[2],
    ((unsigned char *)&addr.sin_addr.s_addr)[3],
    ntohs( addr.sin_port ) );

  //create output socket
  out_sock_fd = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP );
  if( out_sock_fd == -1 )
  {
    close( in_sock_fd );
    td_fmt_free( p_td_fmt );
    free( p_out_buffers );
    free( p_in_buffer );
    printf( "could not create output socket: error: %s\n\n", strerror( errno ) );
    return -1;
  }

  //unblock output socket
  i = 1;
  if( ioctl( out_sock_fd, FIONBIO, &i ) == -1 )
  {
    close( out_sock_fd );
    close( in_sock_fd );
    td_fmt_free( p_td_fmt );
    free( p_out_buffers );
    free( p_in_buffer );
    printf( "could not unblock output socket: error: %s\n\n", strerror( errno ) );
    return -1;
  }

  //bind output socket
  addr.sin_family = AF_INET;
  addr.sin_port = htons( td_out_bind_port );
  addr.sin_addr.s_addr = td_out_bind_addr;
  if( bind( out_sock_fd, (struct sockaddr *)&addr, sizeof( addr ) ) == -1 )
  {
    close( out_sock_fd );
    close( in_sock_fd );
    td_fmt_free( p_td_fmt );
    free( p_out_buffers );
    free( p_in_buffer );
    printf( "could not bind output socket to \"%d.%d.%d.%d:%d\" (udp): error: %s\n\n",
      ((unsigned char *)&addr.sin_addr.s_addr)[0],
      ((unsigned char *)&addr.sin_addr.s_addr)[1],
      ((unsigned char *)&addr.sin_addr.s_addr)[2],
      ((unsigned char *)&addr.sin_addr.s_addr)[3],
      ntohs( addr.sin_port ),
      strerror( errno ) );
    return -1;
  }
  printf( "output socket bound to \"%d.%d.%d.%d:%d\" (udp)\n",
    ((unsigned char *)&addr.sin_addr.s_addr)[0],
    ((unsigned char *)&addr.sin_addr.s_addr)[1],
    ((unsigned char *)&addr.sin_addr.s_addr)[2],
    ((unsigned char *)&addr.sin_addr.s_addr)[3],
    ntohs( addr.sin_port ) );

  //install signal handler to end program
  signal( SIGTERM, end_signal );
  signal( SIGINT, end_signal );
  signal( SIGHUP, end_signal );

  //main loop
  while( ! end )
  {
    //wait for reception of a datagram
    FD_ZERO( &fdset );
    FD_SET( in_sock_fd, &fdset );
    timeout.tv_sec = 0;
    timeout.tv_usec = 100000;
    cnt = select( in_sock_fd + 1, &fdset, NULL, NULL, &timeout );
    if( cnt == -1 && errno != EINTR )
    {
      printf( "select returned error: %s\n\n", strerror( errno ) );
      break;
    }
    //reception of a datagram
    if( cnt > 0 && FD_ISSET( in_sock_fd, &fdset ) )
    {
      //get datagram
      addr_len = sizeof( addr );
      len = recvfrom( in_sock_fd, p_in_buffer, in_buffer_len, 0,
                      (struct sockaddr *)&addr, &addr_len );
      if( len >= 0 )
      {
        if( td_verbose )
          printf( "reception from \"%d.%d.%d.%d:%d\" (udp) (length=%d)\n",
            ((unsigned char *)&addr.sin_addr.s_addr)[0],
            ((unsigned char *)&addr.sin_addr.s_addr)[1],
            ((unsigned char *)&addr.sin_addr.s_addr)[2],
            ((unsigned char *)&addr.sin_addr.s_addr)[3],
            ntohs( addr.sin_port ),
            len );
        //processs datagram
        if( proto_datagram( p_in_buffer, (unsigned int)len, p_td_fmt, p_out_buffers ) )
        {
          //send output data using output socket
          if( td_verbose )
            printf( "outputting data to %d devices (length=%d)\n", td_out_dev_cnt, out_buffer_len );
          in_addr = td_out_ip_base;
          for( dev_no = 0; dev_no < td_out_dev_cnt; dev_no++ )
          {
            //assemble destination address
            addr.sin_family = AF_INET;
            addr.sin_port = htons( td_out_dest_port );
            addr.sin_addr.s_addr = in_addr;
            //send
            if( sendto( out_sock_fd, p_out_buffers[dev_no], out_buffer_len, 0,
                        (struct sockaddr *)&addr, sizeof( addr ) ) != (int)out_buffer_len )
            {
              if( td_verbose )
                printf( "could not send to \"%d.%d.%d.%d:%d\" (udp) (length=%d): error: %s\n",
                  ((unsigned char *)&addr.sin_addr.s_addr)[0],
                  ((unsigned char *)&addr.sin_addr.s_addr)[1],
                  ((unsigned char *)&addr.sin_addr.s_addr)[2],
                  ((unsigned char *)&addr.sin_addr.s_addr)[3],
                  ntohs( addr.sin_port ),
                  out_buffer_len,
                  strerror( errno ) );
            }
            //calculate IP address of next device
            in_addr += td_out_ip_step;
          } //for( dec_no ...
        } //if( proto_datagram( ...
      } //if( len ...
    } // if( cnt > 0 ...
  } //while( ! end )
  printf( "terminated\n\n" );

  //clean up
  close( out_sock_fd );
  close( in_sock_fd );
  td_fmt_free( p_td_fmt );
  free( p_out_buffers );
  free( p_in_buffer );

  return 0;
}
